% This file computes the THD of the sawtooth wave filtered by a low-pass Butterworth filter of order p. 
% The THD is given by \sqrt{2\sum_{n=2}^{\infty} \frac{1}{n^2(1+n^{2p})}}.
% The calculation is carried out in 3 different ways: exact residue-based method, 
% approximate analytic method and brute numerical summation;
% the results are respectively denoted by THD_ex,THD_ap and THD_num.
%
% Author: Iaroslav V. Blagouchine, first version of this file was written in March 2011.  

function [THD_ex,THD_ap,THD_num]=thdswcomputing(p,nmax)

if nargin<1
	p=1;
end

if nargin<2
	nmax=50; % Precision of the numerical method, but also computing time...
end


% Exact analytical method
% Computation of the poles
l=0:2*p-1;
zp=exp(i*pi*(2*l+1)/(2*p));
% Exact analytic computation of theTHD.
for s=1:2*p
	% Product without crossed term
	prpart=zp(s)-zp;
	prpart(s)=1;
	pr(s)=1/prod(prpart);
	% Residue at the point z_s:
	tmp(s)=pr(s)*ctg(pi*zp(s))/zp(s)^2;
end
% Sum of the residues
sumres=sum(tmp);
% The limit g(eps,p)
g=pi^2/3 - 0.5*sign(p-1.5) - 0.5;
% Final THD
THD_ex=real(sqrt(-pi*sumres+g));

% Approximate analytical method
% If using Bernoulli numbers called via: "B_6=mfun('bernoulli(6)',1)"
% Works only for given p, because it is implemented via Maple.
% p=1;
%THD_ap2=sqrt((2*pi)^(2*p+2)*abs(mfun('bernoulli(4)',1))/factorial(2*p+2)-2)
% So I prefer to use Riemann zeta-function. 
THD_ap=sqrt(2*zeta(2*p+2)-2);
	
% Brute numerical method with max number nmax
for k=2:nmax
	f(k,:)=1./(k^2*(1+k^(2*p)));
end
THD_num=sqrt(2*sum(f,1));